class CallManager {
    constructor() {
        this.device = null;
        this.currentCall = null;
        this.isCallActive = false;
        this.timer = null;
        this.setupRetries = 0;
        this.maxRetries = 3;
        this.bindEvents();
        this.initializeDevice();
    }

    async initializeDevice() {
        try {
            console.log('Initializing device...');
            const response = await fetch('?ajax=get_voice_token');
            const data = await response.json();
            
            if (!data.success || !data.token) {
                console.error('Token response:', data);
                throw new Error('Failed to get token');
            }

            console.log('Token received, setting up device');
            
            // Initialize device
            this.device = new Twilio.Device(data.token, {
                debug: true,
                enableRingingState: true,
                allowIncomingWhileBusy: true
            });

            // Device event listeners
            this.device.on('ready', () => {
                console.log('Device is ready');
                this.setupComplete = true;
            });

            this.device.on('error', (error) => {
                console.error('Device error:', error);
                this.handleError(error);
            });

            // Register the device
            await this.device.register();
            console.log('Device registered successfully');

        } catch (error) {
            console.error('Device initialization error:', error);
            this.handleError(error);
        }
    }

    bindEvents() {
        const callButton = document.getElementById('callButton');
        if (callButton) {
            callButton.addEventListener('click', async () => {
                try {
                    if (!this.isCallActive) {
                        await this.startCall();
                    } else {
                        await this.endCall();
                    }
                } catch (error) {
                    console.error('Call button error:', error);
                }
            });
        }
    }

    async requestMicrophoneAccess() {
        try {
            const stream = await navigator.mediaDevices.getUserMedia({
                audio: {
                    echoCancellation: true,
                    noiseSuppression: true,
                    autoGainControl: true
                }
            });
            return stream;
        } catch (error) {
            console.error('Microphone access error:', error);
            throw new Error('Microphone access is required for calls');
        }
    }

    async startCall() {
        try {
            // Get the phone number
            const phoneInput = document.getElementById('contact_phone');
            if (!phoneInput || !phoneInput.value) {
                throw new Error('No phone number available');
            }

            // Format phone number
            let phoneNumber = phoneInput.value.replace(/[^+0-9]/g, '');
            if (!phoneNumber.startsWith('+')) {
                phoneNumber = '+' + phoneNumber;
            }

            console.log('Starting call to:', phoneNumber);

            // Request microphone access
            await this.requestMicrophoneAccess();

            // Show call UI first
            this.showCallUI(phoneNumber);

            // Make the call
            this.currentCall = await this.device.connect({
                params: {
                    To: phoneNumber
                }
            });

            // Set up call event handlers
            this.currentCall.on('accept', () => {
                console.log('Call accepted');
                this.updateCallStatus('Connected');
                this.startCallTimer();
            });

            this.currentCall.on('disconnect', () => {
                console.log('Call disconnected');
                this.handleDisconnect();
            });

            this.isCallActive = true;

            // Update call button
            const callButton = document.getElementById('callButton');
            if (callButton) {
                callButton.innerHTML = '<i class="fas fa-phone-slash"></i> End';
                callButton.classList.remove('bg-green-500');
                callButton.classList.add('bg-red-500');
            }

        } catch (error) {
            console.error('Error making call:', error);
            this.handleError(error);
        }
    }

    endCall() {
        if (this.currentCall) {
            this.currentCall.disconnect();
            this.currentCall = null;
        }

        this.isCallActive = false;
        this.stopCallTimer();

        // Reset call button
        const callButton = document.getElementById('callButton');
        if (callButton) {
            callButton.innerHTML = '<i class="fas fa-phone"></i> Call';
            callButton.classList.remove('bg-red-500');
            callButton.classList.add('bg-green-500');
        }

        Swal.close();
    }

    handleError(error) {
        console.error('Call error:', error);
        this.endCall();
        Swal.fire({
            icon: 'error',
            title: 'Call Error',
            text: error.message || 'Failed to make call',
            timer: 3000,
            showConfirmButton: false
        });
    }

    handleDisconnect() {
        this.currentCall = null;
        this.isCallActive = false;
        this.stopCallTimer();
        setTimeout(() => this.endCall(), 1000);
    }

    showCallUI(phoneNumber) {
        Swal.fire({
            title: '<div class="flex justify-center mb-4"><div class="w-16 h-16 bg-green-500 rounded-full flex items-center justify-center"><i class="fas fa-phone text-white text-2xl"></i></div></div>Connecting...',
            html: `
                <div class="text-lg mb-4">${phoneNumber}</div>
                <div class="flex justify-center space-x-4">
                    <button id="muteButton" class="p-3 rounded-full bg-gray-200 hover:bg-gray-300">
                        <i class="fas fa-microphone"></i>
                    </button>
                    <button id="endCallButton" class="p-3 rounded-full bg-red-500 hover:bg-red-600 text-white">
                        <i class="fas fa-phone-slash"></i>
                    </button>
                    <button id="speakerButton" class="p-3 rounded-full bg-gray-200 hover:bg-gray-300">
                        <i class="fas fa-volume-up"></i>
                    </button>
                </div>
                <div class="mt-4 text-gray-500" id="callTimer">00:00</div>
            `,
            showConfirmButton: false,
            allowOutsideClick: false,
            allowEscapeKey: false,
            didOpen: () => {
                document.getElementById('endCallButton').addEventListener('click', () => {
                    this.endCall();
                });

                document.getElementById('muteButton').addEventListener('click', (e) => {
                    if (this.currentCall) {
                        const icon = e.currentTarget.querySelector('i');
                        const button = e.currentTarget;
                        const isMuted = !this.currentCall.isMuted();
                        
                        this.currentCall.mute(isMuted);
                        
                        if (isMuted) {
                            icon.classList.replace('fa-microphone', 'fa-microphone-slash');
                            button.classList.add('bg-red-200');
                        } else {
                            icon.classList.replace('fa-microphone-slash', 'fa-microphone');
                            button.classList.remove('bg-red-200');
                        }
                    }
                });

                document.getElementById('speakerButton').addEventListener('click', (e) => {
                    e.currentTarget.classList.toggle('bg-blue-200');
                });
            }
        });
    }

    updateCallStatus(status) {
        const titleElement = document.querySelector('.swal2-title');
        if (titleElement) {
            titleElement.innerHTML = `
                <div class="flex justify-center mb-4">
                    <div class="w-16 h-16 bg-green-500 rounded-full flex items-center justify-center">
                        <i class="fas fa-phone text-white text-2xl"></i>
                    </div>
                </div>
                ${status}
            `;
        }
    }

    startCallTimer() {
        this.stopCallTimer();
        let seconds = 0;
        this.timer = setInterval(() => {
            seconds++;
            const minutes = Math.floor(seconds / 60);
            const remainingSeconds = seconds % 60;
            const timerElement = document.getElementById('callTimer');
            if (timerElement) {
                timerElement.textContent = `${String(minutes).padStart(2, '0')}:${String(remainingSeconds).padStart(2, '0')}`;
            }
        }, 1000);
    }

    stopCallTimer() {
        if (this.timer) {
            clearInterval(this.timer);
            this.timer = null;
        }
    }
}