class VoiceCall {
    constructor() {
        this.device = null;
        this.currentConnection = null;
        this.setupComplete = false;
    }

    async init() {
        try {
            // Load Twilio Client JS
            await this.loadTwilioClient();
            
            // Get token
            const response = await fetch('voice-token.php');
            const data = await response.json();
            
            if (!data.token) {
                throw new Error('Failed to get token');
            }

            // Setup Twilio.Device
            this.device = new Twilio.Device(data.token, {
                debug: true,
                codecPreferences: ['opus', 'pcmu']
            });

            // Setup event handlers
            this.device.on('ready', () => {
                this.setupComplete = true;
                console.log('Twilio.Device is ready');
            });

            this.device.on('error', (error) => {
                console.error('Twilio.Device Error:', error);
            });

            this.device.on('connect', (conn) => {
                this.currentConnection = conn;
                this.updateCallUI('connected');
            });

            this.device.on('disconnect', () => {
                this.currentConnection = null;
                this.updateCallUI('disconnected');
            });

            await this.device.register();

        } catch (error) {
            console.error('Error initializing voice call:', error);
            throw error;
        }
    }

    async loadTwilioClient() {
        return new Promise((resolve, reject) => {
            const script = document.createElement('script');
            script.src = 'https://sdk.twilio.com/js/client/releases/1.14.0/twilio.js';
            script.onload = resolve;
            script.onerror = reject;
            document.head.appendChild(script);
        });
    }

    async makeCall(phoneNumber) {
        if (!this.setupComplete) {
            await this.init();
        }

        if (!this.device) {
            throw new Error('Twilio Device not initialized');
        }

        if (this.currentConnection) {
            throw new Error('Already on a call');
        }

        const params = {
            To: phoneNumber
        };

        try {
            this.currentConnection = await this.device.connect({ params });
            this.showCallUI();
        } catch (error) {
            console.error('Error making call:', error);
            throw error;
        }
    }

    endCall() {
        if (this.currentConnection) {
            this.currentConnection.disconnect();
            this.currentConnection = null;
        }
        this.hideCallUI();
    }

    showCallUI() {
        const callUI = document.createElement('div');
        callUI.id = 'call-ui';
        callUI.innerHTML = `
            <div class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
                <div class="bg-white rounded-lg p-6 w-full max-w-sm">
                    <div class="text-center">
                        <div class="mb-4">
                            <div class="w-16 h-16 bg-blue-500 rounded-full mx-auto flex items-center justify-center">
                                <i class="fas fa-phone text-white text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-lg font-semibold mb-2" id="call-status">Calling...</h3>
                        <p class="text-gray-600" id="call-timer">00:00</p>
                        <div class="mt-6 flex justify-center space-x-4">
                            <button id="toggle-mute" class="p-3 rounded-full bg-gray-200 hover:bg-gray-300">
                                <i class="fas fa-microphone"></i>
                            </button>
                            <button id="end-call" class="p-3 rounded-full bg-red-500 hover:bg-red-600 text-white">
                                <i class="fas fa-phone-slash"></i>
                            </button>
                            <button id="toggle-speaker" class="p-3 rounded-full bg-gray-200 hover:bg-gray-300">
                                <i class="fas fa-volume-up"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(callUI);

        // Setup event listeners
        document.getElementById('end-call').addEventListener('click', () => this.endCall());
        document.getElementById('toggle-mute').addEventListener('click', () => this.toggleMute());
        document.getElementById('toggle-speaker').addEventListener('click', () => this.toggleSpeaker());

        this.startTimer();
    }

    hideCallUI() {
        const callUI = document.getElementById('call-ui');
        if (callUI) {
            callUI.remove();
        }
        this.stopTimer();
    }

    updateCallUI(status) {
        const statusEl = document.getElementById('call-status');
        if (statusEl) {
            switch (status) {
                case 'connected':
                    statusEl.textContent = 'Connected';
                    break;
                case 'disconnected':
                    statusEl.textContent = 'Call Ended';
                    setTimeout(() => this.hideCallUI(), 1500);
                    break;
                default:
                    statusEl.textContent = status;
            }
        }
    }

    toggleMute() {
        if (this.currentConnection) {
            const isMuted = !this.currentConnection.isMuted();
            this.currentConnection.mute(isMuted);
            const muteButton = document.getElementById('toggle-mute');
            muteButton.innerHTML = isMuted ? 
                '<i class="fas fa-microphone-slash"></i>' : 
                '<i class="fas fa-microphone"></i>';
            muteButton.classList.toggle('bg-red-200', isMuted);
        }
    }

    toggleSpeaker() {
        // This is a placeholder - actual speaker control depends on browser capabilities
        const speakerButton = document.getElementById('toggle-speaker');
        speakerButton.classList.toggle('bg-blue-200');
    }

    startTimer() {
        let seconds = 0;
        this.timer = setInterval(() => {
            seconds++;
            const minutes = Math.floor(seconds / 60);
            const remainingSeconds = seconds % 60;
            const timerEl = document.getElementById('call-timer');
            if (timerEl) {
                timerEl.textContent = `${minutes.toString().padStart(2, '0')}:${remainingSeconds.toString().padStart(2, '0')}`;
            }
        }, 1000);
    }

    stopTimer() {
        if (this.timer) {
            clearInterval(this.timer);
            this.timer = null;
        }
    }
}
