<?php
class Message {
    private $db;
    private $twilio;

    public function __construct() {
        $this->db = Database::getInstance();
        $config = Config::getInstance();
        
        $this->twilio = new Twilio\Rest\Client(
            $config->get('twilio.account_sid'),
            $config->get('twilio.auth_token')
        );
    }

    public function getContactMessages($contactId) {
        try {
            $sql = "SELECT * FROM messages WHERE contact_id = ? ORDER BY created_at DESC LIMIT 50";
            $stmt = $this->db->getPdo()->prepare($sql);
            $stmt->execute([$contactId]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error getting contact messages: " . $e->getMessage());
            return [];
        }
    }

    public function getNewMessages($contactId, $lastId) {
        try {
            $sql = "SELECT * FROM messages 
                    WHERE contact_id = ? AND id > ? 
                    ORDER BY created_at ASC";
            $stmt = $this->db->getPdo()->prepare($sql);
            $stmt->execute([$contactId, $lastId]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error getting new messages: " . $e->getMessage());
            return [];
        }
    }

    public function send($contactId, $to, $content) {
        try {
            // Format phone number
            if (substr($to, 0, 1) !== '+') {
                $to = '+' . $to;
            }

            // Send via Twilio
            $config = Config::getInstance();
            $message = $this->twilio->messages->create($to, [
                'from' => $config->get('twilio.phone_number'),
                'body' => $content
            ]);

            // Save to database
            $sql = "INSERT INTO messages (contact_id, direction, content, created_at) 
                    VALUES (?, 'outbound', ?, NOW())";
            $stmt = $this->db->getPdo()->prepare($sql);
            $stmt->execute([$contactId, $content]);

            // Get the saved message
            $sql = "SELECT * FROM messages WHERE contact_id = ? ORDER BY id DESC LIMIT 1";
            $stmt = $this->db->getPdo()->prepare($sql);
            $stmt->execute([$contactId]);
            $savedMessage = $stmt->fetch(PDO::FETCH_ASSOC);

            return [
                'success' => true, 
                'sid' => $message->sid,
                'message' => $savedMessage
            ];
        } catch (Exception $e) {
            error_log("Error sending message: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    public function saveInbound($contactId, $content) {
        try {
            $sql = "INSERT INTO messages (contact_id, direction, content, created_at) 
                    VALUES (?, 'inbound', ?, NOW())";
            $stmt = $this->db->getPdo()->prepare($sql);
            $result = $stmt->execute([$contactId, $content]);

            if ($result) {
                // Get the saved message
                $sql = "SELECT * FROM messages WHERE contact_id = ? ORDER BY id DESC LIMIT 1";
                $stmt = $this->db->getPdo()->prepare($sql);
                $stmt->execute([$contactId]);
                return $stmt->fetch(PDO::FETCH_ASSOC);
            }
            return false;
        } catch (Exception $e) {
            error_log("Error saving inbound message: " . $e->getMessage());
            return false;
        }
    }
}