class VoiceCall {
    constructor() {
        this.device = null;
        this.currentConnection = null;
        this.setupComplete = false;
        this.timer = null;
        console.log('VoiceCall initialized');
    }

    async init() {
        try {
            // Request necessary permissions first
            await this.requestPermissions();
            
            console.log('Initializing VoiceCall');
            
            // Get token
            const response = await fetch('?ajax=get_voice_token');
            const data = await response.json();
            
            if (!data.token || !data.success) {
                console.error('Token response:', data);
                throw new Error('Failed to get token: ' + (data.error || 'Unknown error'));
            }
            console.log('Got token successfully');

            // Initialize Device with the new token
            if (this.device) {
                this.device.destroy();
            }

            // Create new device
            this.device = new Twilio.Device();

            // Setup device event listeners
            this.device.on('ready', () => {
                console.log('Device is ready');
                this.setupComplete = true;
            });

            this.device.on('error', (error) => {
                console.error('Device Error:', error);
                this.handleCallError(error);
            });

            this.device.on('incoming', (connection) => {
                console.log('Incoming connection:', connection);
            });

            // Setup the device with the token
            await this.device.setup(data.token, {
                debug: true,
                audioConstraints: {
                    autoGainControl: true,
                    echoCancellation: true,
                    noiseSuppression: true
                }
            });

            console.log('Device setup complete');
            this.setupComplete = true;

        } catch (error) {
            console.error('Error initializing voice call:', error);
            Swal.fire('Error', 'Failed to initialize call device: ' + error.message, 'error');
            throw error;
        }
    }

    async requestPermissions() {
        try {
            console.log('Requesting microphone permission...');
            const stream = await navigator.mediaDevices.getUserMedia({ 
                audio: {
                    echoCancellation: true,
                    noiseSuppression: true,
                    autoGainControl: true
                }
            });
            stream.getTracks().forEach(track => track.stop());
            console.log('Microphone permission granted');
        } catch (error) {
            console.error('Error getting microphone permission:', error);
            throw new Error('Microphone access is required for calls. Please allow microphone access and try again.');
        }
    }

    async makeCall(phoneNumber) {
        console.log('Attempting to make call to:', phoneNumber);
        try {
            // Initialize if not already done
            if (!this.setupComplete || !this.device) {
                await this.init();
            }

            if (this.currentConnection) {
                throw new Error('Already on a call');
            }

            // Clean the phone number
            phoneNumber = phoneNumber.replace(/[^+0-9]/g, '');
            if (!phoneNumber.startsWith('+')) {
                phoneNumber = '+' + phoneNumber;
            }

            // Show UI before making the call
            this.showCallUI();
            console.log('Connecting to:', phoneNumber);

            // Make the call
            this.currentConnection = await this.device.connect({
                params: {
                    To: phoneNumber
                }
            });

            console.log('Connection established, setting up listeners');

            // Set up connection listeners
            this.currentConnection.on('accept', () => {
                console.log('Call accepted');
                this.updateCallUI('connected');
            });

            this.currentConnection.on('disconnect', () => {
                console.log('Call disconnected');
                this.handleCallEnd();
            });

            this.currentConnection.on('error', (error) => {
                console.error('Connection error:', error);
                this.handleCallError(error);
            });

            this.currentConnection.on('warning', (warning) => {
                console.warn('Connection warning:', warning);
            });

        } catch (error) {
            console.error('Call failed:', error);
            this.hideCallUI();
            this.handleCallError(error);
            throw error;
        }
    }

    handleCallError(error) {
        console.error('Call error:', error);
        this.hideCallUI();
        Swal.fire({
            icon: 'error',
            title: 'Call Error',
            text: error.message || 'Failed to complete call',
            footer: 'Please check your connection and try again'
        });
    }

    handleCallEnd() {
        if (this.currentConnection) {
            this.currentConnection = null;
        }
        setTimeout(() => {
            this.hideCallUI();
            Swal.fire({
                icon: 'info',
                title: 'Call Ended',
                timer: 2000,
                showConfirmButton: false
            });
        }, 1000);
    }

    endCall() {
        console.log('Ending call...');
        if (this.currentConnection) {
            this.currentConnection.disconnect();
            this.currentConnection = null;
        }
        this.hideCallUI();
    }

    showCallUI() {
        // Remove existing call UI if present
        this.hideCallUI();

        const callUI = document.createElement('div');
        callUI.id = 'call-ui';
        callUI.innerHTML = `
            <div class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
                <div class="bg-white rounded-lg p-6 w-full max-w-sm">
                    <div class="text-center">
                        <div class="mb-4">
                            <div class="w-16 h-16 bg-blue-500 rounded-full mx-auto flex items-center justify-center">
                                <i class="fas fa-phone text-white text-2xl"></i>
                            </div>
                        </div>
                        <h3 class="text-lg font-semibold mb-2" id="call-status">Calling...</h3>
                        <p class="text-gray-600" id="call-timer">00:00</p>
                        <div class="mt-6 flex justify-center space-x-4">
                            <button id="toggle-mute" class="p-3 rounded-full bg-gray-200 hover:bg-gray-300">
                                <i class="fas fa-microphone"></i>
                            </button>
                            <button id="end-call" class="p-3 rounded-full bg-red-500 hover:bg-red-600 text-white">
                                <i class="fas fa-phone-slash"></i>
                            </button>
                            <button id="toggle-speaker" class="p-3 rounded-full bg-gray-200 hover:bg-gray-300">
                                <i class="fas fa-volume-up"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(callUI);

        // Setup event listeners
        document.getElementById('end-call').addEventListener('click', () => this.endCall());
        document.getElementById('toggle-mute').addEventListener('click', () => this.toggleMute());
        document.getElementById('toggle-speaker').addEventListener('click', () => this.toggleSpeaker());

        this.startTimer();
    }

    hideCallUI() {
        const callUI = document.getElementById('call-ui');
        if (callUI) {
            callUI.remove();
        }
        this.stopTimer();
    }

    updateCallUI(status) {
        const statusEl = document.getElementById('call-status');
        if (statusEl) {
            switch (status) {
                case 'connected':
                    statusEl.textContent = 'Connected';
                    break;
                case 'disconnected':
                    statusEl.textContent = 'Call Ended';
                    setTimeout(() => this.hideCallUI(), 1500);
                    break;
                default:
                    statusEl.textContent = status;
            }
        }
    }

    toggleMute() {
        if (this.currentConnection) {
            const isMuted = !this.currentConnection.isMuted();
            this.currentConnection.mute(isMuted);
            const muteButton = document.getElementById('toggle-mute');
            muteButton.innerHTML = isMuted ? 
                '<i class="fas fa-microphone-slash"></i>' : 
                '<i class="fas fa-microphone"></i>';
            muteButton.classList.toggle('bg-red-200', isMuted);
        }
    }

    toggleSpeaker() {
        const speakerButton = document.getElementById('toggle-speaker');
        speakerButton.classList.toggle('bg-blue-200');
    }

    startTimer() {
        let seconds = 0;
        this.timer = setInterval(() => {
            seconds++;
            const minutes = Math.floor(seconds / 60);
            const remainingSeconds = seconds % 60;
            const timerEl = document.getElementById('call-timer');
            if (timerEl) {
                timerEl.textContent = `${minutes.toString().padStart(2, '0')}:${remainingSeconds.toString().padStart(2, '0')}`;
            }
        }, 1000);
    }

    stopTimer() {
        if (this.timer) {
            clearInterval(this.timer);
            this.timer = null;
        }
    }
}