<?php
class Call {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function create($data) {
        try {
            $sql = "INSERT INTO calls (contact_id, direction, status, duration, start_time, end_time, recording_url, twilio_sid, notes) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $stmt = $this->db->getPdo()->prepare($sql);
            return $stmt->execute([
                $data['contact_id'],
                $data['direction'],
                $data['status'],
                $data['duration'] ?? 0,
                $data['start_time'],
                $data['end_time'] ?? null,
                $data['recording_url'] ?? null,
                $data['twilio_sid'] ?? null,
                $data['notes'] ?? null
            ]);
        } catch (PDOException $e) {
            error_log("Error creating call record: " . $e->getMessage());
            return false;
        }
    }

    public function getByContactId($contactId) {
        try {
            $sql = "SELECT * FROM calls WHERE contact_id = ? ORDER BY start_time DESC";
            $stmt = $this->db->getPdo()->prepare($sql);
            $stmt->execute([$contactId]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Error getting calls for contact: " . $e->getMessage());
            return [];
        }
    }

    public function updateStatus($twilioSid, $status, $duration = null) {
        try {
            $sql = "UPDATE calls 
                    SET status = ?, 
                        duration = ?, 
                        end_time = CASE WHEN ? IN ('completed', 'failed', 'busy', 'no-answer') THEN NOW() ELSE end_time END 
                    WHERE twilio_sid = ?";
            
            $stmt = $this->db->getPdo()->prepare($sql);
            return $stmt->execute([
                $status,
                $duration,
                $status,
                $twilioSid
            ]);
        } catch (PDOException $e) {
            error_log("Error updating call status: " . $e->getMessage());
            return false;
        }
    }

    public function getCallStats($contactId) {
        try {
            $sql = "SELECT 
                        COUNT(*) as total_calls,
                        SUM(CASE WHEN direction = 'outbound' THEN 1 ELSE 0 END) as outbound_calls,
                        SUM(CASE WHEN direction = 'inbound' THEN 1 ELSE 0 END) as inbound_calls,
                        SUM(duration) as total_duration,
                        AVG(duration) as avg_duration
                    FROM calls 
                    WHERE contact_id = ?";
            
            $stmt = $this->db->getPdo()->prepare($sql);
            $stmt->execute([$contactId]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Error getting call stats: " . $e->getMessage());
            return [
                'total_calls' => 0,
                'outbound_calls' => 0,
                'inbound_calls' => 0,
                'total_duration' => 0,
                'avg_duration' => 0
            ];
        }
    }
}