<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Contacts Management</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="assets/js/voice.js"></script>
<script>
    const voiceCall = new VoiceCall();

    async function startCall() {
        try {
            const phoneNumber = document.getElementById('contact_phone').value;
            await voiceCall.makeCall(phoneNumber);
        } catch (error) {
            console.error('Error starting call:', error);
            Swal.fire('Error', 'Failed to start call. Please try again.', 'error');
        }
    }

    // Update the call button click handler
    document.getElementById('callButton').addEventListener('click', startCall);
</script>
</head>
<body class="bg-gray-100">
    <div class="min-h-screen">
        <!-- Header -->
        <header class="bg-white shadow">
            <div class="max-w-7xl mx-auto px-4 py-6">
                <div class="flex justify-between items-center">
                    <h1 class="text-2xl font-bold text-gray-900">Contact Management</h1>
                    <button onclick="openAddModal()" class="bg-blue-500 text-white px-4 py-2 rounded-lg hover:bg-blue-600 flex items-center gap-2">
                        <i class="fas fa-plus"></i> Add Contact
                    </button>
                </div>
            </div>
        </header>

        <!-- Main Content -->
        <main class="max-w-7xl mx-auto px-4 py-6">
            <!-- Search and Filter -->
            <div class="bg-white rounded-lg shadow mb-6 p-4">
                <div class="flex gap-4">
                    <div class="flex-1">
                        <input type="text" 
                               id="searchInput" 
                               placeholder="Search contacts..." 
                               class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <select id="filterStatus" class="px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <option value="">All Contacts</option>
                        <option value="recent">Recent Messages</option>
                        <option value="no_messages">No Messages</option>
                    </select>
                </div>
            </div>

            <!-- Contacts Grid -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                <?php foreach ($contacts as $contact): ?>
                    <div class="bg-white rounded-lg shadow hover:shadow-md transition-shadow">
                        <div class="p-6">
                            <div class="flex justify-between items-start">
                                <div class="flex items-center gap-4">
                                    <div class="w-12 h-12 bg-blue-500 rounded-full flex items-center justify-center text-white text-xl font-semibold">
                                        <?= strtoupper(substr($contact['name'], 0, 1)) ?>
                                    </div>
                                    <div>
                                        <h3 class="text-lg font-semibold"><?= htmlspecialchars($contact['name']) ?></h3>
                                        <p class="text-gray-600"><?= htmlspecialchars($contact['phone']) ?></p>
                                        <?php if($contact['email']): ?>
                                            <p class="text-gray-500 text-sm"><?= htmlspecialchars($contact['email']) ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="flex gap-2">
                                    <button onclick="editContact(<?= $contact['id'] ?>)" 
                                            class="text-gray-400 hover:text-gray-600">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button onclick="deleteContact(<?= $contact['id'] ?>)" 
                                            class="text-gray-400 hover:text-red-600">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Quick Actions -->
                            <div class="mt-4 pt-4 border-t flex justify-between">
                                <a href="?page=contacts&action=view&id=<?= $contact['id'] ?>" 
                                   class="text-blue-500 hover:text-blue-600 flex items-center gap-2">
                                    <i class="fas fa-comment"></i>
                                    Message
                                </a>
                                <button onclick="startCall('<?= htmlspecialchars($contact['phone']) ?>')"
                                        class="text-green-500 hover:text-green-600 flex items-center gap-2">
                                    <i class="fas fa-phone"></i>
                                    Call
                                </button>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </main>
    </div>

    <!-- Add/Edit Contact Modal -->
    <div id="contactModal" class="fixed inset-0 bg-black bg-opacity-50 hidden flex items-center justify-center">
        <div class="bg-white rounded-lg p-6 w-full max-w-md">
            <h2 id="modalTitle" class="text-xl font-semibold mb-4">Add New Contact</h2>
            <form id="contactForm" onsubmit="saveContact(event)">
                <input type="hidden" id="contactId" value="">
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Name</label>
                        <input type="text" id="contactName" required
                               class="mt-1 block w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Phone</label>
                        <input type="tel" id="contactPhone" required
                               class="mt-1 block w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                               placeholder="+1234567890">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Email</label>
                        <input type="email" id="contactEmail"
                               class="mt-1 block w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Notes</label>
                        <textarea id="contactNotes"
                                  class="mt-1 block w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"></textarea>
                    </div>
                </div>
                <div class="mt-6 flex justify-end gap-3">
                    <button type="button" onclick="closeModal()"
                            class="px-4 py-2 text-gray-600 hover:text-gray-800">
                        Cancel
                    </button>
                    <button type="submit"
                            class="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600">
                        Save Contact
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
    function openAddModal() {
        document.getElementById('modalTitle').textContent = 'Add New Contact';
        document.getElementById('contactId').value = '';
        document.getElementById('contactName').value = '';
        document.getElementById('contactPhone').value = '';
        document.getElementById('contactEmail').value = '';
        document.getElementById('contactNotes').value = '';
        document.getElementById('contactModal').classList.remove('hidden');
    }

    function closeModal() {
        document.getElementById('contactModal').classList.add('hidden');
    }

    async function editContact(id) {
        try {
            const response = await fetch(`?ajax=get_contact&id=${id}`);
            const data = await response.json();
            
            if (data.success) {
                document.getElementById('modalTitle').textContent = 'Edit Contact';
                document.getElementById('contactId').value = data.contact.id;
                document.getElementById('contactName').value = data.contact.name;
                document.getElementById('contactPhone').value = data.contact.phone;
                document.getElementById('contactEmail').value = data.contact.email || '';
                document.getElementById('contactNotes').value = data.contact.notes || '';
                document.getElementById('contactModal').classList.remove('hidden');
            }
        } catch (error) {
            console.error('Error fetching contact:', error);
            Swal.fire('Error', 'Failed to load contact details', 'error');
        }
    }

    async function saveContact(event) {
        event.preventDefault();
        
        const formData = new FormData();
        formData.append('id', document.getElementById('contactId').value);
        formData.append('name', document.getElementById('contactName').value);
        formData.append('phone', document.getElementById('contactPhone').value);
        formData.append('email', document.getElementById('contactEmail').value);
        formData.append('notes', document.getElementById('contactNotes').value);

        try {
            const response = await fetch('?ajax=save_contact', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                Swal.fire('Success', 'Contact saved successfully', 'success')
                    .then(() => window.location.reload());
            } else {
                Swal.fire('Error', result.error || 'Failed to save contact', 'error');
            }
        } catch (error) {
            console.error('Error saving contact:', error);
            Swal.fire('Error', 'Failed to save contact', 'error');
        }
    }

    async function deleteContact(id) {
        const result = await Swal.fire({
            title: 'Are you sure?',
            text: "You won't be able to revert this!",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Yes, delete it!'
        });

        if (result.isConfirmed) {
            try {
                const response = await fetch('?ajax=delete_contact', {
                    method: 'POST',
                    body: JSON.stringify({ id: id }),
                    headers: {
                        'Content-Type': 'application/json'
                    }
                });
                
                const result = await response.json();
                
                if (result.success) {
                    Swal.fire('Deleted!', 'Contact has been deleted.', 'success')
                        .then(() => window.location.reload());
                } else {
                    Swal.fire('Error', result.error || 'Failed to delete contact', 'error');
                }
            } catch (error) {
                console.error('Error deleting contact:', error);
                Swal.fire('Error', 'Failed to delete contact', 'error');
            }
        }
    }

    // Search functionality
    document.getElementById('searchInput').addEventListener('input', function(e) {
        const searchTerm = e.target.value.toLowerCase();
        const contacts = document.querySelectorAll('.grid > div');
        
        contacts.forEach(contact => {
            const name = contact.querySelector('h3').textContent.toLowerCase();
            const phone = contact.querySelector('p').textContent.toLowerCase();
            const email = contact.querySelector('p:nth-child(2)')?.textContent.toLowerCase() || '';
            
            if (name.includes(searchTerm) || phone.includes(searchTerm) || email.includes(searchTerm)) {
                contact.style.display = '';
            } else {
                contact.style.display = 'none';
            }
        });
    });

    function startCall(phone) {
        // Will implement in next step
        Swal.fire('Coming Soon', 'Call feature will be implemented soon!', 'info');
    }
    </script>
</body>
</html>