<?php
require 'vendor/autoload.php';
use Ratchet\Server\IoServer;
use Ratchet\Http\HttpServer;
use Ratchet\WebSocket\WsServer;
use React\EventLoop\Factory;

class ChatWebSocket implements MessageComponentInterface {
    protected $clients;
    protected $subscriptions = [];

    public function __construct() {
        $this->clients = new \SplObjectStorage;
    }

    public function onOpen(ConnectionInterface $conn) {
        $this->clients->attach($conn);
        echo "New connection! ({$conn->resourceId})\n";
    }

    public function onMessage(ConnectionInterface $from, $msg) {
        $data = json_decode($msg, true);
        
        if ($data && isset($data['type'])) {
            switch ($data['type']) {
                case 'subscribe':
                    if (isset($data['contactId'])) {
                        $this->subscriptions[$from->resourceId] = $data['contactId'];
                        echo "Client {$from->resourceId} subscribed to contact {$data['contactId']}\n";
                    }
                    break;
            }
        }
    }

    public function onClose(ConnectionInterface $conn) {
        $this->clients->detach($conn);
        unset($this->subscriptions[$conn->resourceId]);
        echo "Connection {$conn->resourceId} has disconnected\n";
    }

    public function onError(ConnectionInterface $conn, \Exception $e) {
        echo "An error has occurred: {$e->getMessage()}\n";
        $conn->close();
    }

    public function broadcast($contactId, $data) {
        foreach ($this->clients as $client) {
            if (isset($this->subscriptions[$client->resourceId]) 
                && $this->subscriptions[$client->resourceId] == $contactId) {
                $client->send(json_encode($data));
            }
        }
    }
}

$loop = Factory::create();
$webSocket = new ChatWebSocket();

$server = IoServer::factory(
    new HttpServer(
        new WsServer(
            $webSocket
        )
    ),
    8080
);

$server->run();