class ChatManager {
    constructor(contactId) {
        this.contactId = contactId;
        this.messages = [];
        this.lastMessageId = 0;
        this.messageContainer = document.getElementById('chat-messages');
        this.messageForm = document.getElementById('message-form');
        this.messageInput = document.getElementById('message-input');
        this.isPolling = false;
        this.pollingInterval = 3000; // 3 seconds
        this.init();
    }

    async init() {
        try {
            await this.loadMessages();
            this.bindEvents();
            this.startPolling();
            this.scrollToBottom();
            console.log('Chat initialized for contact:', this.contactId);
        } catch (error) {
            console.error('Chat initialization error:', error);
            this.handleError('Failed to initialize chat');
        }
    }

    bindEvents() {
        // Message form submission
        if (this.messageForm) {
            this.messageForm.addEventListener('submit', async (e) => {
                e.preventDefault();
                await this.sendMessage();
            });
        }

        // Handle enter key
        if (this.messageInput) {
            this.messageInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    this.messageForm.dispatchEvent(new Event('submit'));
                }
            });
        }

        // Load more messages on scroll to top
        if (this.messageContainer) {
            this.messageContainer.addEventListener('scroll', () => {
                if (this.messageContainer.scrollTop === 0) {
                    this.loadOlderMessages();
                }
            });
        }
    }

    startPolling() {
        if (!this.isPolling) {
            this.isPolling = true;
            this.poll();
            setInterval(() => this.poll(), this.pollingInterval);
        }
    }

    async poll() {
        try {
            const response = await fetch(`?ajax=check_new_messages&contact_id=${this.contactId}&last_id=${this.lastMessageId}`);
            const data = await response.json();

            if (data.success && data.messages.length > 0) {
                data.messages.forEach(message => {
                    this.handleNewMessage(message);
                });
            }
        } catch (error) {
            console.error('Polling error:', error);
        }
    }

    async loadMessages() {
        try {
            const response = await fetch(`?ajax=get_messages&contact_id=${this.contactId}`);
            const data = await response.json();

            if (data.success) {
                this.messages = data.messages;
                if (this.messages.length > 0) {
                    this.lastMessageId = Math.max(...this.messages.map(m => m.id));
                }
                this.renderMessages();
                this.scrollToBottom();
            }
        } catch (error) {
            console.error('Error loading messages:', error);
            this.handleError('Failed to load messages');
        }
    }

    async sendMessage() {
        const content = this.messageInput.value.trim();
        if (!content) return;

        try {
            const formData = new FormData();
            formData.append('contact_id', this.contactId);
            formData.append('message', content);
            formData.append('to', document.getElementById('contact_phone').value);

            // Show temporary message
            const tempMessage = {
                id: 'temp-' + Date.now(),
                direction: 'outbound',
                content: content,
                created_at: new Date().toISOString(),
                status: 'sending'
            };
            this.addMessage(tempMessage);

            const response = await fetch('?ajax=send_message', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();
            if (result.success) {
                // Remove temporary message and add real one
                this.removeMessage(tempMessage.id);
                this.handleNewMessage(result.message);
                this.messageInput.value = '';
            } else {
                this.removeMessage(tempMessage.id);
                throw new Error(result.error || 'Failed to send message');
            }
        } catch (error) {
            console.error('Error sending message:', error);
            this.handleError('Failed to send message');
        }
    }

    handleNewMessage(message) {
        if (message.id > this.lastMessageId) {
            this.lastMessageId = message.id;
            this.messages.push(message);
            this.renderMessage(message);
            
            if (message.direction === 'inbound') {
                this.playNotificationSound();
                this.showNotification(message);
            }
            
            this.scrollToBottom();
        }
    }

    addMessage(message) {
        this.messages.push(message);
        this.renderMessage(message);
        this.scrollToBottom();
    }

    removeMessage(messageId) {
        const messageEl = document.querySelector(`[data-message-id="${messageId}"]`);
        if (messageEl) {
            messageEl.remove();
        }
        this.messages = this.messages.filter(m => m.id !== messageId);
    }

    renderMessages() {
        if (!this.messageContainer) return;
        
        this.messageContainer.innerHTML = '';
        this.messages.forEach(message => {
            this.renderMessage(message);
        });
    }

    renderMessage(message) {
        if (!this.messageContainer) return;

        const messageEl = document.createElement('div');
        messageEl.className = `message ${message.direction}`;
        messageEl.dataset.messageId = message.id;
        
        messageEl.innerHTML = `
            <div class="message-content">
                ${this.escapeHtml(message.content)}
            </div>
            <div class="message-time">
                ${this.formatDate(message.created_at)}
                ${message.status === 'sending' ? '<span class="sending-indicator">Sending...</span>' : ''}
            </div>
        `;

        this.messageContainer.appendChild(messageEl);
    }

    scrollToBottom() {
        if (this.messageContainer) {
            this.messageContainer.scrollTop = this.messageContainer.scrollHeight;
        }
    }

    playNotificationSound() {
        try {
            const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBkCa2/LDciUFLYHO8tiJNwgZabvt559NEAxQp+PwtmMcBj+b2/LDciUFLYHO8diJNwgZabzu559NEAxQp+PwtmMcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+Pwt');
            audio.play().catch(e => console.log('Audio play failed:', e));
        } catch (error) {
            console.log('Notification sound failed:', error);
        }
    }

    showNotification(message) {
        if ('Notification' in window && Notification.permission === 'granted') {
            new Notification('New Message', {
                body: message.content
            });
        }
    }

    formatDate(dateStr) {
        const date = new Date(dateStr);
        const today = new Date();
        const yesterday = new Date();
        yesterday.setDate(yesterday.getDate() - 1);

        if (date.toDateString() === today.toDateString()) {
            return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        } else if (date.toDateString() === yesterday.toDateString()) {
            return 'Yesterday ' + date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        } else {
            return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        }
    }

    escapeHtml(unsafe) {
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }

    handleError(message) {
        console.error(message);
        Swal.fire({
            icon: 'error',
            title: 'Error',
            text: message,
            timer: 3000,
            showConfirmButton: false
        });
    }
}